#!/bin/sh
#% the next line restarts using wish \
exec wish "$0" "$@"
#
# All of the source code, object code, data files, image files,
# documentation, and any other associated RenderWrapper files are
# considered copyrighted property of Robert W. Lindeman. You are free to
# distribute the code as you like, provided you don't receive any
# compensation for any programs incorporating any or all of the code
# contained in RenderWrapper.
#
# Any images you create with the aid of RenderWrapper are, of course,
# your own property, and can be distributed as you see fit.
#
# Copyright (c) 1998 Robert W. Lindeman
#
# -------------------------------------------------------------------------
# Script      : render -r<renderer> -a<arguments> -s<source> -f<rib>
#
# Description : This script starts a rendering task based on the
#               parameters passed to it.
#
# Parameters  : The argument "-r<renderer>" is one of:
#                   rendrib
#                   rgl
#               The argument "-a<arguments>" is a string of argumants
#               to pass to the renderer.
#               The argument "-s<source>" is one of:
#                   file
#                   program
#               The argument "-f<rib>" is either a RIB file, or a program
#               name including arguments.
#
# Returns     : void.
#
# -------------------------------------------------------------------------
# Function    : InitProg { argc argv }
#
# Description : This procedure initializes all the global variables for the
#               program.
#
# Parameters  : argc : Number of command-line arguments.
#               argv : Vector of command-line arguments.
#
# Returns     : void.

proc InitProg { argc argv }  {
  global ProgName
  global BarFG
  global BarBG
  global BarFrameFG
  global BarFrameBG
  global BarFont
  global MessageFG
  global MessageBG
  global MessageFont
  global DefaultFG
  global DefaultBG
  global DefaultAFG
  global DefaultABG
  global DefaultTrough
  global DefaultMenuFG
  global DefaultMenuBG
  global DefaultMenuAFG
  global DefaultMenuABG
  global DefaultMenuSBG
  global DefaultEntryFG
  global DefaultEntryBG
  global CRTFont
  global CRAFont
  global CRCFont
  global TitleFG
  global TitleBG
  global DefaultFont
  global PromptFont
  global EntryFont
  global HelpFont
  global HelpHeaderFont
  global BarString
  global BarLength
  global SmallScreen
  global XRes
  global YRes
  global Version
  global Release
  global IconName
  global tcl_platform
  global CopyrightLine
  global IconImage

  # State-Variables
  global Renderer
  global RIBSource
  global CmdLine

  global Rendrib
  global RGL

  set Rendrib(ResX)          ""
  set Rendrib(ResY)          ""
  set Rendrib(SamplesX)      ""
  set Rendrib(SamplesY)      ""
  set Rendrib(VarV)          ""
  set Rendrib(VarMin)        ""
  set Rendrib(VarMax)        ""
  set Rendrib(StartFrame)    ""
  set Rendrib(EndFrame)      ""
  set Rendrib(RadioSteps)    ""
  set Rendrib(RadioSamples)  ""
  set Rendrib(CropX1)        ""
  set Rendrib(CropY1)        ""
  set Rendrib(CropX2)        ""
  set Rendrib(CropY2)        ""
  set Rendrib(Safe)           0
  set Rendrib(Display)        0
  set Rendrib(DispInterlace) ""
  set Rendrib(PosX)          ""
  set Rendrib(PosY)          ""
  set Rendrib(Ascii)          0
  set Rendrib(Verbose)        0
  set Rendrib(Stats)          0
  set Rendrib(Silent)         0
  set Rendrib(Rayserver)      0

  set RGL(ResX)              ""
  set RGL(ResY)              ""
  set RGL(PosX)              ""
  set RGL(PosY)              ""
  set RGL(CropX1)            ""
  set RGL(CropY1)            ""
  set RGL(CropX2)            ""
  set RGL(CropY2)            ""
  set RGL(StartFrame)        ""
  set RGL(EndFrame)          ""
  set RGL(SyncN)             ""
  set RGL(NoWait)             0
  set RGL(Stats)              0
  set RGL(1Buffer)            0
  set RGL(Sketch)             0
  set RGL(DumpRGB)            0
  set RGL(Display)            1
  set RGL(Verbose)            0
  set RGL(Wireframe)          0
  set RGL(IgnoreLights)       0
  set RGL(Detail)            ""
  set RGL(FrameCommand)      ""
  set RGL(SoundFile)         ""

  set ProgName           "RenderWrapper"
  set BarFG              yellow
  set BarBG              blue
  set BarFrameFG         black
  set BarFrameBG         lightgrey
  set BarFont            { Courier 14 }
  set MessageFG          white
  set MessageBG          black
  set MessageFont        { Courier 12 }
  set DefaultFG          black
  set DefaultBG          lightgrey
  set DefaultAFG         black
  set DefaultABG         snow2
  set DefaultTrough      grey
  set DefaultMenuFG      $DefaultFG
  set DefaultMenuBG      $DefaultBG
  set DefaultMenuAFG     $DefaultFG
  set DefaultMenuABG     $DefaultBG
  set DefaultMenuSBG     $DefaultBG
  set DefaultEntryFG     $DefaultFG
  set DefaultEntryBG     $DefaultBG
  set TitleFG            yellow
  set TitleBG            maroon
  set DefaultFont        { Helvetica 12 bold }
  set PromptFont         { Courier 12 }
  set EntryFont          { Helvetica 12 }
  set HelpFont           { Helvetica 12 }
  set HelpHeaderFont     { Helvetica 16 bold }
  set BarString          "                    %3d%%                     "
  set BarLength          45
  set SmallScreen        "FALSE"
  # Get the dimensions of the screen.
  set XRes [ winfo screenwidth . ]
  set YRes [ winfo screenheight . ]
  set Version            "1.0"
  set Release            ".0"
  set IconName           "$ProgName ${Version}${Release}"
  set CopyrightLine "Copyright (c) 1999 Lindeman Systems International, Ltd. - All Rights Reserved"
  set CmdLine            ""

  # Initialize the global params.
  set CurRenderer        ""
  set CurRIBSource       ""
  set CurRIBFile         ""
  set CurRIBProg         ""

  # Set the platform-dependent stuff.
  switch -regexp -- $tcl_platform(platform)  { \
    ^macintosh$  {
       ;# Nothing to do here.
    } \
    ^windows$  {
      # Close the extra console that Win Tcl creates.
      console hide

      set DefaultFG          SystemButtonText
      set DefaultBG          SystemButtonFace
      set DefaultAFG         SystemButtonText
      set DefaultABG         SystemButtonFace
      set DefaultTrough      "#C3C3C3"
      set DefaultMenuFG      SystemMenuText
      set DefaultMenuBG      SystemMenu
      set DefaultMenuAFG     SystemHighlightText
      set DefaultMenuABG     SystemHighlight
      set DefaultMenuSBG     SystemMenuText
      set DefaultFont        { Helvetica 12 bold }
      # Fonts for the "about" window.
      set CRTFont            { Helvetica 12 bold }
      set CRAFont            { Helvetica 8 }
      set CRCFont            { Helvetica 8 }

      # Set the icon label for the program.
      wm iconname   . $IconName
    } \
    ^unix$  {
      set DefaultFG          black
      set DefaultBG          "#D9D9D9"
      set DefaultAFG         black
      set DefaultABG         "#ECECEC"
      set DefaultTrough      "#C3C3C3"
      set DefaultMenuFG      black
      set DefaultMenuBG      "#D9D9D9"
      set DefaultMenuAFG     black
      set DefaultMenuABG     "#ECECEC"
      set DefaultMenuSBG     "#B03060"
      set DefaultFont        { Helvetica 12 bold }
      # Fonts for the "about" window.
      set CRTFont            { Helvetica 18 bold }
      set CRAFont            { Helvetica 12 }
      set CRCFont            { Helvetica 10 }

      # Set the icon label for the program.
      wm iconname . $IconName
    } \
  }

# Set the icon.
#wm iconbitmap . @$IconImage

  # Check for the year.
#  if { $argc > 0 }  {
#    foreach arg $argv  {
#      switch -regexp -- $arg  { \
#        default  {
#          Usage
#        } \
#      }
#    }
#  }
};

# -------------------------------------------------------------------------
# Function    : CleanUpAndExit { askUser }
#
# Description : This procedure exits the program. The user is asked to
#               acknowledge if askUser is 1.
#
# Parameters  : askUser : FLAG: Ask user before exiting?
#
# Returns     : void.

proc CleanUpAndExit { askUser }  {
  if { $askUser == 1 }  {
    ;
    exit
  } else  {
    exit
  }
}

# -------------------------------------------------------------------------
# Function    : Usage { }
#
# Description : This procedure prints a usage message and exits.
#
# Parameters  : void.
#
# Returns     : void.

proc Usage { }  {
  puts "use: render -r(rendrib|rgl) -a<args> -s(file|program) -f<file>"
  exit
}

# -------------------------------------------------------------------------
# Function    : BuildStatusWindow { parentTag }
#
# Description : This procedure constructs the geometry for the progress
#               bar and status window, and returns it.
#
# Parameters  : parentTag : Tag to use as parent.
#
# Returns     : void.

proc BuildStatusWindow { parentTag }  {
  global BarFG
  global BarBG
  global BarString
  global BarLength
  global DefaultFG
  global DefaultBG
  global DefaultAFG
  global DefaultABG
  global DefaultTrough
  global DefaultFont
  global BarFont
  global BarFrameFG
  global BarFrameBG
  global MessageFG
  global MessageBG
  global MessageFont
  global MessText
  global BarText
  global ImageFile
  global Renderer

  set progressFrame ${parentTag}.progressFrame
  frame ${progressFrame} -relief raised -bg $DefaultBG
  pack ${progressFrame}
  frame ${progressFrame}.messFrame -relief raised -bg $DefaultBG
  frame ${progressFrame}.buttonFrame -relief raised -bg $DefaultBG
  label ${progressFrame}.messTitle -text "Messages from Renderer:" -anchor w \
        -relief raised -bd 2 -font $DefaultFont
  set MessText ${progressFrame}.messText
  text  ${MessText} -relief raised -bd 2 -font $MessageFont \
        -width 80 -height 12 \
        -yscrollcommand "${progressFrame}.messScroll set" \
        -foreground $MessageFG -background $MessageBG
  scrollbar ${progressFrame}.messScroll -command "${MessText} yview" \
            -troughcolor $DefaultTrough -bg $DefaultBG \
            -activebackground $DefaultABG \
            -highlightthickness 0 -orient vertical
  button ${progressFrame}.doneButton -text "Dismiss" -anchor c \
            -relief raised -bd 4 -font $DefaultFont \
            -command { CleanUpAndExit 0 }

  pack ${progressFrame}.doneButton -side top -pady 4 \
       -in ${progressFrame}.buttonFrame
  pack ${progressFrame}.buttonFrame -side bottom \
       -in ${progressFrame}

  if { $Renderer == "rendrib" }  {
    frame ${progressFrame}.barFrame1 -relief raised -bg $DefaultBG
    label ${progressFrame}.barTitle -text "Rendering Progress:" -anchor c \
          -relief raised -bd 2 -font $DefaultFont
    set ImageFile \"\"
    label ${progressFrame}.imageFile -textvariable ImageFile -anchor c \
          -relief raised -bd 1 -font $DefaultFont -fg $BarBG
    frame ${progressFrame}.barFrame2 -relief groove -bg $BarFrameBG -bd 2

    # Create the bar.
    set BarText ${progressFrame}.barText
    text ${BarText} -height 1 -width $BarLength -font $BarFont \
         -foreground $BarFrameFG -background $BarFrameBG
    ${BarText} insert end [ format $BarString 0 ]
    ${BarText} tag add bar 1.0 1.0
    ${BarText} tag configure bar -relief raised -border 2 -font $BarFont \
         -foreground $BarFG -background $BarBG

    pack ${progressFrame}.barFrame1 -side bottom -fill x -side bottom \
         -pady 4 -in ${progressFrame}
    pack ${progressFrame}.barTitle -side top -fill x \
         -in ${progressFrame}.barFrame1
    pack ${progressFrame}.imageFile -side top -pady 2 \
         -in ${progressFrame}.barFrame1
    pack ${progressFrame}.barFrame2 -side bottom \
         -in ${progressFrame}.barFrame1
    pack ${BarText} -in ${progressFrame}.barFrame2 -pady 10
  }
  
  pack ${progressFrame}.messFrame -side left \
       -in ${progressFrame}
  pack ${progressFrame}.messTitle -side top -fill x \
       -in ${progressFrame}.messFrame
  pack ${MessText} -side left \
       -in ${progressFrame}.messFrame
  pack ${progressFrame}.messScroll -side right -fill y \
       -in ${progressFrame}.messFrame
};

# -------------------------------------------------------------------------
# Function    : UpdateStatusWindow { pipeId parentTag }
#
# Description : This procedure updates the progress bar and status window
#               for a rendering session.
#
# Parameters  : pipeId    : ID of pipe to read from.
#               parentTag : Tag to use as parent.
#
# Returns     : void.

proc UpdateStatusWindow { pipeId parentTag }  {
  global BarString
  global BarLength
  global MessText
  global BarText
  global ImageFile

  set line [gets $pipeId] ;# *NOT* read !!!!
  set pct "-1"
  if { ( [ regexp {^Done computing ([0-9]+)%} $line dummy pct ] != 0 ) || \
       ( [ regexp {^Done computing image\.} $line ] != 0 ) }  {
    if { $pct == "-1" }  {
      set pct 100
      ${MessText} insert end $line
      ${MessText} insert end "\n"
      ${MessText} insert end "\n"
      ${MessText} yview end
    }
    ${BarText} delete 1.0 end
    ${BarText} insert end [ format $BarString $pct ]
    ${BarText} tag add bar \
                1.0 [ format "1.%d" [ expr ( $pct * $BarLength ) / 100 ] ]
  } else  {
    if { [ regexp {^Rendering \"([0-9a-zA-Z_\.\-\\#\(\)\+\/]+)\"} \
                                        $line dummy imageFile ] != 0 }  {
      ${MessText} insert end $line
      ${MessText} insert end "\n"
      ${MessText} yview end
      set ImageFile [ format "\"%s\"" $imageFile ]
      set pct 0
      ${BarText} delete 1.0 end
      ${BarText} insert end [ format $BarString $pct ]
      ${BarText} tag add bar \
                1.0 [ format "1.%d" [ expr ( $pct * $BarLength ) / 100 ] ]
    } else  {
      if { [ string length $line ] != 0 }  {
        ${MessText} insert end $line
        ${MessText} insert end "\n"
        ${MessText} yview end
      }
    }
  }
  update idletasks
#  puts $line
};

# -------------------------------------------------------------------------
# This is the main procedure.
#

set RW_Path_Env RENDERWRAPPER_PATH

# Check for the necessary environment variables.
if { ![ info exists env($RW_Path_Env) ] }  {
  puts ""
  puts "Error:"
  puts " The environment variable \"$RW_Path_Env\" is not set."
  puts " Please set this variable equal to the full-path of the"
  puts " directory where the file \"render.tcl\" resides."
  puts ""
  tk_messageBox -icon error -type ok -default ok \
        -title "Error!" -parent . \
        -message "Error: The environment variable \"$RW_Path_Env\" is not set! \
Please set this variable equal to the full-path of the directory where the file \"render.tcl\" resides, and restart."
  exit
}

# This line allows Tcl to find the common procedures.
set auto_path [ linsert $auto_path 0 [ pwd ] ]
set auto_path [ linsert $auto_path 0 [ file join [ pwd ] lib ] ]
set auto_path [ linsert $auto_path 0 $env($RW_Path_Env) ]
set auto_path [ linsert $auto_path 0 [ file join $env($RW_Path_Env) lib ] ]

# Initialize global variables.
InitProg $argc $argv

# Change the title.
wm title . $ProgName
wm geometry . "+0+100"

set Renderer   ""
set RenderArgs ""
set RIBArgs    ""
set Source     ""

# Check for the year.
if { $argc > 0 }  {
  foreach arg $argv  {
    switch -regexp -- $arg  {
      ^\-a  {
        set RenderArgs [ string range $arg 2 end ]
      }
      ^\-f  {
        set RIBArgs [ string range $arg 2 end ]
      }
      ^\-r(rendrib|rgl)$  {
        scan $arg "\-r%s" Renderer
      }
      ^\-s(file|program)$  {
        scan $arg "\-s%s" Source
      }
      default  {
        puts "use: render -r(rendrib|rgl) -a<args> -s(file|program) -f<file>"
        exit
      }
    }
  }
}

# Check that all args have been entered.
if { ( [ string length $Renderer ] == 0 ) || \
     ( [ string length $Source ] == 0 ) || \
     ( [ string length $RIBArgs ] == 0 ) }  {
  Usage
}

#puts "r = $Renderer"
#puts "a = $RenderArgs"
#puts "s = $Source"
#puts "f = $RIBArgs"

set parentTag ".m"

frame $parentTag -bg $DefaultBG
pack $parentTag

# Build the progress bar.
BuildStatusWindow $parentTag

switch -regexp -- $tcl_platform(platform)  { \
  ^macintosh$  {
     ;# Nothing to do here.
  } \
  ^windows$  {
    if { $Source == "file" }  {
      set pipeId [ open \
          "|$Renderer $RenderArgs $RIBArgs" r ]
    } else  {
      set pipeId [ open \
          "|$RIBArgs | $Renderer $RenderArgs" r ]
    }
  } \
  ^unix$  {
    if { $Source == "file" }  {
      set pipeId [ open \
          "|sh -c { exec $Renderer $RenderArgs $RIBArgs 2>&1 }" r ]
    } else  {
      set pipeId [ open \
          "|sh -c { exec sh -c $RIBArgs | $Renderer $RenderArgs 2>&1 }" r ]
    }
  } \
}

fconfigure $pipeId -buffering line
fileevent $pipeId readable [ list UpdateStatusWindow $pipeId $parentTag ]

# -------------------------------------------------------------------------
